/**
  Copyright (c) 2010 Freescale Semiconductor
  
  \file  	  GALLO.c
  \brief	  This is the Graphics Memory Allocation Driver
  \brief	  Allocates space in memory for layers on xPC560xS (SPECTRUM)
  \author	  Freescale Semiconductor
  \author	  Automotive Systems Solutions Engineering
  \author	  IM, b06623
  \version	  2.0
  \revision	  $Revision: 58 $
  \date  	  $Date: 2010-11-22 20:11:42 -0600 (Mon, 22 Nov 2010) $
  
  * History:  10/July/2008  - Initial Version (IM)
              23/March/2009 - Added fields to main structure. Check on defragment method (IM)
              11/May/2009   - Release (IM)
              17/May/2010	- Update to support optimize use of DMA driver (IM)
              16/Aug/2010	- Added context switching (IM)
              18/Nov/2010	- fix GALLO_freememory() bug, invalid data when stackindex = 0
              				- added define for dcu lite memory partition
	      
  * MISRA VIOLATIONS:
	- [ MISRA 11.2 ]
	- [ MISRA 16.6 ] 
	- [ MISRA 16.9 ]

* Copyright (c) 2010, Freescale, Inc.  All rights reserved.
*  
*  
* No part of this document must be reproduced in any form - including copied,
* transcribed, printed or by any electronic means - without specific written
* permission from Freescale Semiconductor.
*
  
*/

#include	 "..\HAL\GALLO.h"


/* pointers for switching context */
/* (Defines to keep code intact) */
GALLO_StackType *GALLO_Stack;
uint16_t		*GALLO_StackIndex_ptr;
#define GALLO_StackIndex 	(*GALLO_StackIndex_ptr)
uint8_t  		*GALLO_DeFragmented_ptr;	
#define	GALLO_DeFragmented 	(*GALLO_DeFragmented_ptr)
uint8_t			*GALLO_InternalState_ptr;
#define GALLO_InternalState (*GALLO_InternalState_ptr)
uint32_t		GALLO_memstart;
uint32_t		GALLO_memsize;
#define  	GALLO_MEMSTART	GALLO_memstart
#define  	GALLO_MEMSIZE	GALLO_memsize
#define __GALLO_SIZE  0x40000
#define __GALLO_BEGIN 0x3F040000



GALLO_CallbackType GALLO_Callback;

/* for MPC5606S DCU */
GALLO_StackType	GALLO_Stack1[GALLO_SLOTS];

uint8_t 		GALLO_Ctx = 55;
uint16_t		GALLO_StackIndex1;
uint8_t  		GALLO_DeFragmented1;
uint8_t			GALLO_InternalState1;
GALLO_CallbackType GALLO_Callback1;

#ifdef DCU_LITE
/* for MPC5606S DCULite */
GALLO_StackType	GALLO_Stack2[GALLO_SLOTS];

uint16_t		GALLO_StackIndex2;
uint8_t  		GALLO_DeFragmented2;
uint8_t			GALLO_InternalState2;
GALLO_CallbackType GALLO_Callback2;
#endif

#define GALLO_IDLE		(55u)
#define GALLO_BUSY		(1u)
#define GALLO_FAILED	(2u)

static void GALLO_DFTask(uint8_t eDMAChannel);

void GALLO_SetContextDCU()
{
	GALLO_Ctx = 0;
	GALLO_Stack = GALLO_Stack1;
	GALLO_StackIndex_ptr = &GALLO_StackIndex1;
	GALLO_DeFragmented_ptr = &GALLO_DeFragmented1;
	GALLO_InternalState_ptr = &GALLO_InternalState1;
	GALLO_Callback = GALLO_Callback1;
	GALLO_MEMSTART = GALLO_MEMSTART0;
	GALLO_MEMSIZE = GALLO_MEMSIZE0;
}

#ifdef DCU_LITE
void GALLO_SetContextDCULITE()
{
	GALLO_Ctx = 1;
	GALLO_Stack = GALLO_Stack2;
	GALLO_StackIndex_ptr = &GALLO_StackIndex2;
	GALLO_DeFragmented_ptr = &GALLO_DeFragmented2;
	GALLO_InternalState_ptr = &GALLO_InternalState2;	
	GALLO_Callback = GALLO_Callback2;
	GALLO_MEMSTART = GALLO_MEMSTART1;
	GALLO_MEMSIZE = GALLO_MEMSIZE1;
}
#endif

/**
* \brief	GALLO_Init - Initializes the GALLO driver control variables
* \author	IM, b06623
* \param	void
* \return	void
* \todo
*/
void GALLO_Init( void  )
{
    uint16_t i;

	if(GALLO_Ctx == 55)
	{
		GALLO_SetContextDCU();
	}	
    
    for( i = 0; i < GALLO_SLOTS; i++ )
    {
		GALLO_Stack[i].Size	= 0;
    }
        
    GALLO_StackIndex   = 0;
    GALLO_DeFragmented = 1;
    GALLO_InternalState = GALLO_IDLE;
    
    /* VIOLATION TO THE FOLLOWING RULES (16.9-1) */
    /* [ MISRA 16.9 ] [ MISRA 11.2 ] */
    /* Rationale: It is required to assign a values to this function that will be used as callback */
    /* the null pointer value indicates the function has not been initialized */
    GALLO_Callback1 = NULL_PTR;
}

/**
* \brief	GALLO_TotalFreeMemory - Returns the total free memory.
* \brief	Not all free memory can be used, it should be purged before that.
* \author	IM, b06623
* \param	void
* \return	uint32_t - returns the amount of free bytes.
* \todo
*/
uint32_t GALLO_TotalFreeMemory(void)
{
    uint16_t i;
    uint32_t UsedMemory;

    UsedMemory = GALLO_MEMSIZE;
    
    for( i = 0u; i < GALLO_StackIndex; i++ )
    {
		UsedMemory -= GALLO_Stack[ i ].Size;
    }
    return (UsedMemory);
}


uint32_t GALLO_FreeMemory(void)
{
	if(GALLO_StackIndex == 0)
	{
		return GALLO_MEMSIZE;
	}
	else
	{ 
    	return (GALLO_MEMSIZE + GALLO_MEMSTART - GALLO_Stack[GALLO_StackIndex - 1].DynamicAdd - GALLO_Stack[GALLO_StackIndex - 1].Size);
	}
}

uint8_t GALLO_MemoryDeFragmented(void)
{
    return (GALLO_DeFragmented);
}
		

/**
* \brief	GALLO_GetSize - Returns the size of the given entity. If not found, returns zero.
* \author	IM, b06623
* \param	uint8_t Entity, Entity is the index of the allocated element
* \return	uint32_t - returns the amount of allocated bytes for that entity
* \todo
*/
uint32_t GALLO_GetSize(uint8_t Entity)
{
    uint16_t i;
    uint32_t size;

    size = 0;
    for( i = 0; i < GALLO_StackIndex; i++ )
    {
		if(GALLO_Stack[ i ].Entity == Entity)
		{
		    size = GALLO_Stack[ i ].Size;
		    break;
		}
    }
    return size;
}

/**
* \brief	GALLO_GetLinkAddress - Returns the configured link address.
* \author	IM, b06623
* \param	uint8_t Entity, Entity is the index of the allocated element
* \return	uint32_t* - returns a pointer to the address that is using the allocated memory.
* \todo
*/
uint32_t* GALLO_GetLinkAddress(uint8_t Entity)
{
    uint16_t i;
    uint32_t *ptr;
    
    /* VIOLATION TO THE FOLLOWING RULES  [ MISRA 11.2 ]  */
    /* Rationale: It is required to assign a values to this pointer */
    /* the null pointer value indicates the function has not been initialized */
    ptr = NULL_PTR;
    
    for( i = 0u; i < GALLO_StackIndex; i++ )
    {
		if(GALLO_Stack[ i ].Entity == Entity)
		{
	    	ptr = GALLO_Stack[ i ].Address;
	    	break;
		}
    }
    return ptr;
}

/**
* \brief	GALLO_GetDynamicAddress - Returns the address where the memory is allocated
* \author	IM, b06623
* \param	uint8_t Entity, Entity is the index of the allocated element
* \return	uint32_t - returns the starting address of the allocated section 
* \todo
*/
uint32_t GALLO_GetDynamicAddress(uint8_t Entity)
{
    uint16_t i;
    uint32_t add;

    add = 0;
    
    for( i = 0u; i < GALLO_StackIndex; i++ )
    {
	if( ( GALLO_Stack[ i ].Entity == Entity ) && ( GALLO_Stack[ i ].Size != 0u ) )
	{
	    add = GALLO_Stack[ i ].DynamicAdd;
	    break;
	}
    }
    return add;
}

/**
 * \brief	GALLO_Allocate - Allocates one entity in the graphics memory.
 * \author	IM, b06623
 * \param	uint8_t Entity, Entity is the index of the element that will be allocated. 
 * \param	uint32_t Size, is the number of bytes to allocate.
 * \param	uint32_t *AddressPointer, pointer to a 32 bits variable which holds the address where is the start of the information.
 * \return	GALLO_ErrorType, Ok if the operations is completed correctly. Otherwise check errortypes. 
 * \todo
 */
GALLO_ErrorType GALLO_Allocate(uint8_t Entity, uint32_t Size, uint32_t * AddressPointer)
{
    uint16_t	i;
    GALLO_ErrorType error;

    error = GALLO_ERROR_OK;

    /* Check if stack is not full, if there is enough free memory or if entity is no already allocated */
    if(GALLO_StackIndex >= GALLO_SLOTS)
    {
		error = GALLO_ERROR_FULL;
    }
    else if( GALLO_FreeMemory() < Size )
    {
		error = GALLO_ERROR_FULL;
    }
    else if(GALLO_InternalState == GALLO_BUSY)
    {
		error = GALLO_ERROR_BUSY;
    }
    else if(GALLO_InternalState != GALLO_IDLE)
    {
		error = GALLO_ERROR_FAILED;
    }
    else if(Size == 0u)
    {
		error = GALLO_ERROR_FAILED;
    }     
    else
    {
		/* Look for the entity in the stack */
		for( i = 0u; i < GALLO_StackIndex; i++ )
		{
		    if( ( GALLO_Stack[ i ].Entity == Entity ) && ( GALLO_Stack[ i ].Size != 0 ) )
		    {
				error = GALLO_ERROR_ALLOCATED;
		    }
		}
    }

    if(error == GALLO_ERROR_OK)
    {
		/* Round the size to 64 bits, for optimal eDMA transfer*/
		if( ( Size % 8u ) != 0u )
		{
		    Size = Size + 8u - ( Size % 8u );
		}
		GALLO_Stack[ GALLO_StackIndex ].Entity	 	= Entity;
		GALLO_Stack[ GALLO_StackIndex ].Size	 	= Size;
		GALLO_Stack[ GALLO_StackIndex ].Address		= AddressPointer;
		if(GALLO_StackIndex == 0)
		{
		    GALLO_Stack[ GALLO_StackIndex ].DynamicAdd	= GALLO_MEMSTART;
		}
		else
		{
		    GALLO_Stack[ GALLO_StackIndex ].DynamicAdd	= GALLO_Stack[ GALLO_StackIndex - 1].DynamicAdd + GALLO_Stack[ GALLO_StackIndex - 1].Size;
		}

		if(GALLO_Stack[ GALLO_StackIndex ].Address != NULL_PTR)
		{
		    *GALLO_Stack[ GALLO_StackIndex ].Address = GALLO_Stack[ GALLO_StackIndex ].DynamicAdd;  
		}

		GALLO_StackIndex++;
    }

    return error;
}


/**
* \brief	GALLO_DeAllocate - Inactivates one section of allocated memory.
* \author	IM, b06623
* \param	uint8_t Entity, Entity is the index of the element that will be deallocated. 
* \return	GALLO_ErrorType, Ok if the operations is completed correctly. Otherwise check errortypes.
* \return	Possible results: GALLO_ERROR_OK, GALLO_ERROR_NOTALLOCATED.
* \todo
*/
GALLO_ErrorType GALLO_DeAllocate(uint8_t Entity)
{
    uint16_t	i;
    GALLO_ErrorType error;

    error = GALLO_ERROR_NOTALLOCATED;

    if(GALLO_InternalState == GALLO_BUSY)
    {
		error = GALLO_ERROR_BUSY;
    }
    else if(GALLO_InternalState != GALLO_IDLE)
    {
		error = GALLO_ERROR_FAILED;
    }
    else
    {
		for( i = 0u; i < GALLO_StackIndex; i++ )
		{
		    if( ( GALLO_Stack[ i ].Entity == Entity ) && ( GALLO_Stack[ i ].Size != 0 ) )
		    {
				GALLO_Stack[ i ].Size = 0;
				if(i == ( GALLO_StackIndex - 1))
				{
			    	GALLO_StackIndex--;
				}
				else
				{
			    	GALLO_DeFragmented = 0;
				}
				error = GALLO_ERROR_OK;
				break;
		    }
		}
	}

    return error;
}

/**
* \brief	GALLO_DeFragment - Cleans the gaps between memory which has been deallocated
* \author	IM, b06623
* \param	uint8_t eDMAChannel, The eDMAChannel to use when allocating memory.
* \return	GALLO_ErrorType, Ok if the operations is completed correctly. Otherwise check errortypes.
* \return	Possible results: GALLO_ERROR_OK, GALLO_ERROR_DMAHWBUSY, GALLO_ERROR_DMACHBUSY. 
* \todo
*/
GALLO_ErrorType GALLO_DeFragment(uint8_t eDMAChannel, GALLO_CallbackType callback)
{
    DMA_ErrorType		error;
    GALLO_ErrorType		error1;
    //DMA_ChannelCfgType	cfg1;

    if(GALLO_DeFragmented != 0u )
    {
		error1 = GALLO_ERROR_OK;
		if(callback != NULL_PTR)
		{
	    	callback();
		}
    }
    else if(GALLO_InternalState == GALLO_BUSY)
    {
		error1 = GALLO_ERROR_BUSY;
    }
    else if(GALLO_InternalState != GALLO_IDLE)
    {
		error1 = GALLO_ERROR_FAILED;
    }     
    else
    {
    	/* set TCD all to 0 */
	    error = DMA_InitChannel( eDMAChannel );
	    if( error != DMA_ERROR_OK )
	    {
			error1 = GALLO_ERROR_DMACHBUSY;
	    }
	    else
	    {
			GALLO_InternalState = GALLO_BUSY;
			DMA_DestinationOffset(eDMAChannel) = 4;
			DMA_SourceOffset(eDMAChannel) = 4;
			DMA_SetMajorLoopCount(eDMAChannel,1);
			DMA_SourceSize( eDMAChannel )      = 2u;
			DMA_DestinationSize( eDMAChannel ) = 2u;

			/* VIOLATION TO [ MISRA 16.9 ] See 16.9-1 */
			GALLO_Callback = callback;
			GALLO_DFTask(eDMAChannel);
		    
	    }
    }
    return error1;
}

 
/* Recursive call to defragment block by block */ 
static void GALLO_DFTask(uint8_t eDMAChannel)
{ 
    uint16_t	i;
    uint16_t	j;

    /* Perform defragmentation */
    /* 1.- Search for inactive memory section */
    i = 0;
    while( (GALLO_Stack[ i ].Size != 0u) && (i != GALLO_StackIndex) )
    {
		i++;
    }

    /* There are inactive layers */
    if(i != GALLO_StackIndex)
    {
		/* 2.- Search for the next active memory section */
		j = (uint16_t)(i + 1u);  
		while( (GALLO_Stack[ j ].Size == 0u) && (j != GALLO_StackIndex) )
		{
		    j++;
		}
		/* the next active entity was found */
		if(j != GALLO_StackIndex)
		{
		    /* 3.- Move entity J to I */
		    DMA_SourceAddress( eDMAChannel ) = (uint32_t)GALLO_Stack[ j ].DynamicAdd;
		    DMA_DestinationAddress( eDMAChannel ) = (uint32_t)GALLO_Stack[ i ].DynamicAdd;
		    DMA_MinorLoopCount( eDMAChannel )	  = (uint32_t)GALLO_Stack[ j ].Size;
		    

		    /* Update the bottom new active entity (size non-zero) */
		    GALLO_Stack[ i ].Size = GALLO_Stack[ j ].Size;
		    GALLO_Stack[ i ].Entity = GALLO_Stack[ j ].Entity;
		    GALLO_Stack[ i ].Address = GALLO_Stack[ j ].Address;
		    if(GALLO_Stack[ i ].Address != NULL_PTR)
		    {
				*GALLO_Stack[ i ].Address = GALLO_Stack[ i ].DynamicAdd;
		    }
	 
		    /* Update Upper Entities */
		    /* Size of the upper entities are zero */
		    /* Series of inactive entities are ignored */
		    
		    /* Topper layer, must be inactive */
		    GALLO_Stack[ j ].Size = 0u;
		    /* Consecutive layer, must be inactive */
		    GALLO_Stack[ i+1 ].Size = 0u;
		    GALLO_Stack[ i+1 ].DynamicAdd = GALLO_Stack[ i ].DynamicAdd + GALLO_Stack[ i ].Size;	

		    DMA_BWC(eDMAChannel) = 0;
		    /* Start the transfer */
		    /* VIOLATION TO THE FOLLOWING RULES */
		    /* [ MISRA 16.2 ] Functions shall not call themselves, either directly or indirectly */
		    /* Rationale:  In order to obtain effiency, it needs to be called recursively */
		    /* VIOLATION TO [ MISRA 16.9 ] Rationale: Tied to ratinale above */ 
		    DMA_SetCallback(eDMAChannel,GALLO_DFTask);
		    DMA_Start(eDMAChannel);
		}
		else
		{
		    /*Uninit*/
		    DMA_UnInitChannel(eDMAChannel);
		    /*Callback */
		    /* VIOLATION TO [ MISRA 16.9 ] It is needed to identify null pointers */  
		    if(GALLO_Callback != NULL_PTR)
		    {
				GALLO_Callback();
		    }
		    GALLO_StackIndex = i;
		    GALLO_DeFragmented = 1u;
		    GALLO_InternalState = GALLO_IDLE;
		}
    }
    /* if no more inactive layers */
    /* Defragmentation has been completed! */
    else
    {
		/*Uninit*/
		DMA_UnInitChannel(eDMAChannel);
		/*Callback */
		/* VIOLATION TO [ MISRA 16.9 ] It is needed to identify null pointers */  
		if(GALLO_Callback != NULL_PTR)
		{
	    	GALLO_Callback();
		}
		GALLO_DeFragmented = 1u;
		GALLO_InternalState = GALLO_IDLE;
    }
}